<?php
/* --------------------------------------------------------------
   TrackingCode.php 2020-04-16
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2020 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Admin\TrackingCode\Model;

use DateTime;
use Exception;
use Gambio\Admin\TrackingCode\Interfaces\TrackingCode as TrackingCodeInterface;
use Gambio\Admin\TrackingCode\Interfaces\TrackingCodeId;
use Gambio\Admin\TrackingCode\Model\TrackingCodeId as TrackingCodeIdModel;
use Gambio\Core\Language\LanguageId;
use InvalidArgumentException;
use Webmozart\Assert\Assert;

/**
 * Class TrackingCode
 *
 * @package Gambio\Admin\TrackingCode\Model
 */
class TrackingCode implements TrackingCodeInterface
{
    /**
     * @var TrackingCodeId
     */
    private $id;
    
    /**
     * @var OrderId
     */
    private $orderId;
    
    /**
     * @var string
     */
    private $code;
    
    /**
     * @var LanguageId
     */
    private $languageId;
    
    /**
     * @var ParcelServiceDetails
     */
    private $parcelServiceDetails;
    
    /**
     * @var DateTime
     */
    private $createdOn;
    
    
    /**
     * TrackingCode constructor.
     *
     * @param TrackingCodeId       $id
     * @param OrderId              $orderId
     * @param string               $code
     * @param LanguageId           $languageId
     * @param ParcelServiceDetails $parcelServiceDetails
     * @param DateTime             $createdOn
     */
    private function __construct(
        TrackingCodeId $id,
        OrderId $orderId,
        string $code,
        LanguageId $languageId,
        ParcelServiceDetails $parcelServiceDetails,
        DateTime $createdOn
    ) {
        $this->id                   = $id;
        $this->orderId              = $orderId;
        $this->code                 = $code;
        $this->languageId           = $languageId;
        $this->parcelServiceDetails = $parcelServiceDetails;
        $this->createdOn            = $createdOn;
    }
    
    
    /**
     * @param TrackingCodeId       $id
     * @param OrderId              $orderId
     * @param string               $code
     * @param LanguageId           $languageId
     * @param ParcelServiceDetails $parcelServiceDetails
     * @param DateTime             $createdOn
     *
     * @return TrackingCode
     */
    public static function create(
        TrackingCodeId $id,
        OrderId $orderId,
        string $code,
        LanguageId $languageId,
        ParcelServiceDetails $parcelServiceDetails,
        DateTime $createdOn
    ): TrackingCode {
        Assert::notWhitespaceOnly($code, 'Code can not be empty.');
        
        return new self($id, $orderId, $code, $languageId, $parcelServiceDetails, $createdOn);
    }
    
    
    /**
     * @inheritDoc
     */
    public function id(): ?int
    {
        return $this->id->value();
    }
    
    
    /**
     * @inheritDoc
     */
    public function orderId(): int
    {
        return $this->orderId->value();
    }
    
    
    /**
     * @inheritDoc
     */
    public function code(): string
    {
        return $this->code;
    }
    
    
    /**
     * @inheritDoc
     */
    public function parcelServiceId(): int
    {
        return $this->parcelServiceDetails->id();
    }
    
    
    /**
     * @inheritDoc
     */
    public function languageId(): int
    {
        return $this->languageId->value();
    }
    
    
    /**
     * @inheritDoc
     */
    public function parcelServiceName(): string
    {
        return $this->parcelServiceDetails->name();
    }
    
    
    /**
     * @inheritDoc
     */
    public function url(): string
    {
        return $this->parcelServiceDetails->url();
    }
    
    
    /**
     * @inheritDoc
     */
    public function comment(): string
    {
        return $this->parcelServiceDetails->comment();
    }
    
    
    /**
     * @inheritDoc
     */
    public function createdOn(string $format = 'Y-m-d H:i:s'): string
    {
        return $this->createdOn->format($format);
    }
    
    
    /**
     * @inheritDoc
     */
    public function jsonSerialize()
    {
        return [
            'id'            => $this->id(),
            'orderId'       => $this->orderId(),
            'code'          => $this->code(),
            'languageId'    => $this->languageId(),
            'parcelService' => $this->parcelServiceDetails,
            'createdOn'     => $this->createdOn(),
        ];
    }
    
    
    /**
     * @param array $data
     *
     * @return TrackingCode
     */
    public static function createFromArray(array $data): TrackingCode
    {
        try {
            $createdOn = new DateTime($data['createdOn']);
        } catch (Exception $e) {
            throw new InvalidArgumentException('Invalid datetime format for creation date provided.');
        }
        
        return self::create(TrackingCodeIdModel::create($data['id'] ?? null),
                            OrderId::create($data['orderId']),
                            $data['code'],
                            \Gambio\Core\Language\Model\LanguageId::create($data['languageId']),
                            ParcelServiceDetails::createFromArray($data['parcelService']),
                            $createdOn);
    }
}